<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Vehicle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ProfileController extends Controller
{
    /**
     * Show user profile
     */
    public function index()
    {
        // Check if user is logged in
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Please login to view your profile.');
        }

        $user = Auth::user();
        $user->load([
            'vehicles', 
            'ridesAsDriver.route.fromCity', 
            'ridesAsDriver.route.toCity', 
            'bookingsAsPassenger.ride.route.fromCity', 
            'bookingsAsPassenger.ride.route.toCity'
        ]);

        $stats = [
            'total_rides' => $user->ridesAsDriver->count() + $user->bookingsAsPassenger->count(),
            'as_driver' => $user->ridesAsDriver->count(),
            'as_passenger' => $user->bookingsAsPassenger->count(),
            'rating' => $user->rating ?? 4.5,
        ];

        $tab = request('tab', 'profile');

        return view('profile', compact('user', 'stats', 'tab'));
    }

    /**
     * Update profile information
     */
    public function update(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'user_type' => 'required|in:passenger,driver,both',
        ]);

        $user = Auth::user();
        
        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'user_type' => $request->user_type,
        ]);

        return redirect()->route('profile')->with('success', 'Profile updated successfully!');
    }

    /**
     * Upload profile picture
     */
    public function uploadAvatar(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $request->validate([
            'avatar' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $user = Auth::user();

        // Delete old avatar if exists
        if ($user->avatar) {
            Storage::disk('public')->delete($user->avatar);
        }

        // Store new avatar
        $path = $request->file('avatar')->store('avatars', 'public');
        $user->update(['avatar' => $path]);

        return redirect()->route('profile')->with('success', 'Profile picture updated!');
    }

    /**
     * Submit KYC verification
     */
    public function submitKyc(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $request->validate([
            'cnic' => 'required|string|max:15',
            'cnic_front' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            'cnic_back' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $user = Auth::user();

        // Store CNIC images
        $frontPath = $request->file('cnic_front')->store('kyc', 'public');
        $backPath = $request->file('cnic_back')->store('kyc', 'public');

        // Update user with KYC data (store as JSON in verification_document)
        $kycData = json_encode([
            'cnic' => $request->cnic,
            'cnic_front' => $frontPath,
            'cnic_back' => $backPath,
            'submitted_at' => now()->toDateTimeString(),
            'status' => 'pending'
        ]);

        $user->update([
            'verification_document' => $kycData,
        ]);

        return redirect()->route('profile', ['tab' => 'kyc'])->with('success', 'KYC documents submitted for verification!');
    }

    /**
     * Add a new vehicle
     */
    public function addVehicle(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $request->validate([
            'make' => 'required|string|max:100',
            'model' => 'required|string|max:100',
            'year' => 'required|integer|min:1990|max:' . (date('Y') + 1),
            'color' => 'required|string|max:50',
            'license_plate' => 'required|string|max:20',
            'seats' => 'required|integer|min:1|max:8',
        ]);

        $user = Auth::user();

        Vehicle::create([
            'driver_id' => $user->id,
            'make' => $request->make,
            'model' => $request->model,
            'year' => $request->year,
            'color' => $request->color,
            'license_plate' => $request->license_plate,
            'seats' => $request->seats,
            'is_verified' => false,
        ]);

        return redirect()->route('profile', ['tab' => 'vehicles'])->with('success', 'Vehicle added successfully!');
    }

    /**
     * Delete a vehicle
     */
    public function deleteVehicle($id)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $vehicle = Vehicle::findOrFail($id);
        
        // Make sure user owns this vehicle
        if ($vehicle->driver_id !== Auth::id()) {
            return redirect()->route('profile', ['tab' => 'vehicles'])->with('error', 'Unauthorized action.');
        }
        
        $vehicle->delete();

        return redirect()->route('profile', ['tab' => 'vehicles'])->with('success', 'Vehicle removed!');
    }
}
