<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Ride extends Model
{
    protected $fillable = [
        'driver_id',
        'vehicle_id',
        'route_id',
        'departure_datetime',
        'arrival_datetime',
        'available_seats',
        'price_per_seat',
        'full_car_price',
        'ride_status',
        'pickup_point',
        'drop_point',
        'amenities',
        'ride_preferences',
    ];

    protected function casts(): array
    {
        return [
            'departure_datetime' => 'datetime',
            'arrival_datetime' => 'datetime',
            'available_seats' => 'integer',
            'price_per_seat' => 'decimal:2',
            'full_car_price' => 'decimal:2',
            'amenities' => 'array',
            'ride_preferences' => 'array',
        ];
    }

    // Relationships

    /**
     * Get the driver of this ride
     */
    public function driver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'driver_id');
    }

    /**
     * Get the vehicle used for this ride
     */
    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Get the route for this ride
     */
    public function route(): BelongsTo
    {
        return $this->belongsTo(Route::class);
    }

    /**
     * Get all bookings for this ride
     */
    public function bookings(): HasMany
    {
        return $this->hasMany(Booking::class);
    }

    // Scopes

    /**
     * Scope for upcoming rides only
     */
    public function scopeUpcoming($query)
    {
        return $query->where('departure_datetime', '>', now())
                    ->where('ride_status', 'scheduled');
    }

    /**
     * Scope for rides with available seats
     */
    public function scopeAvailable($query)
    {
        return $query->where('available_seats', '>', 0)
                    ->where('ride_status', 'scheduled');
    }

    /**
     * Scope for completed rides
     */
    public function scopeCompleted($query)
    {
        return $query->where('ride_status', 'completed');
    }

    // Helper Methods

    /**
     * Check if ride is bookable
     */
    public function isBookable(): bool
    {
        return $this->ride_status === 'scheduled' 
            && $this->available_seats > 0 
            && $this->departure_datetime > now();
    }

    /**
     * Get booked seats count
     */
    public function getBookedSeatsAttribute(): int
    {
        return $this->bookings()->where('booking_status', '!=', 'cancelled')->sum('seats_booked');
    }

    /**
     * Get remaining seats
     */
    public function getRemainingSeatsAttribute(): int
    {
        return max(0, $this->vehicle->total_seats - $this->booked_seats);
    }

    /**
     * Get formatted departure time
     */
    public function getDepartureTimeAttribute(): string
    {
        return $this->departure_datetime->format('M d, Y h:i A');
    }

    /**
     * Check if amenity is available
     */
    public function hasAmenity(string $amenity): bool
    {
        return in_array($amenity, $this->amenities ?? []);
    }
}
